<?php
/**
 * Uploaded.to API Class
 * 
 *  Untersttzung von:
 *  - hochladen, herunterladen, Backup, lschen und umbenennen von Dateien 
 *  - erstellen, lschen und umbenennen von Ordnern 
 *  - Zuordnung von Dateien in Ordnern. 
 *  - Remote Uploads
 *  - Free und Premium User Support
 *   
 * @author Julius Fischer
 * @copyright 2011 Julius Fischer
 * 
 * @license GNU LGPL <br>
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * @link http:\\www.it-gecko.de
 * @version 0.3
 *
 */
class Uploaded
{
	protected $user_id = '';
	protected $user_pw = '';
	protected $auth_str = '';
	protected $cookie_str = '';
	protected $login_status = false;
	protected $acc_status = false; //false = free | true = premium
	
	public function __construct($user_id = '', $user_pw = '')
	{
		if($user_id && $user_pw)
			$this->login($user_id, $user_pw);
	}
	
	protected function init_login()
	{
		$res = $this->get_curl('http://uploaded.to/io/login', 'id='.$this->user_id.'&pw='.$this->user_pw, false, true);
		
		if(preg_match('#.'.preg_quote('{err:').'#si', $res))
		{
			$this->login_status = false;
			return false;
		}
		
		if(preg_match('#PHPSESSID\=[a-zA-Z0-9]+;#si', $res, $machtes))
			$this->cookie_str = $machtes[0];
		
		if(preg_match('#login\=.*?;#si', $res, $machtes))
			$this->cookie_str .= $machtes[0];
		
		if(preg_match('#auth\=[a-zA-Z0-9];#si', $res, $machtes))
			$this->cookie_str .= $machtes[0];
		
		$this->login_status = true;
		
		return true;
	}
	
	/**
	 * Prft auf erfolgreichen Login
	 * 
	 * @return bool true = erfolgreich eingeloggt | false = Login fehlgeschlagen
	 */
	public function login_check()
	{
		//return preg_match('#'.preg_quote('<a href="logout">Logout</a>').'#si', $this->get_curl('http://uploaded.to'));
		
		return $this->login_status;
	}
	
	/**
	 * Login
	 * 
	 * @param int $user_id	Benutzer-ID
	 * @param string $user_pw Benutzerpasswort
	 * 
	 * @return bool true = erfolgreich eingeloggt | false = login fehlgeschlagen
	 */
	public function login($user_id, $user_pw)
	{
		$this->user_id 	= (int)$user_id;
		$this->user_pw 	= $user_pw;
		$this->auth_str	= '&id='.$this->user_id.'&pw='.sha1($this->user_pw);
		
		return $this->init_login();
	}
	
	/**
	 * Setzt die Logindaten fr Uploaded.to Premium Account
	 * 
	 * @param int $user_id	Benutzer-ID
	 * @param string $user_pw Benutzerpasswort
	 * 
	 * @return bool true = erfolgreich eingeloggt | false = login fehlgeschlagen
	 * 
	 * @deprecated Benutze login()
	 */
	public function set_login($user_id, $user_pw)
	{
		/*
		$this->user_id 	= (int)$user_id;
		$this->user_pw 	= $user_pw;		
		$this->auth_str	= '&id='.$this->user_id.'&pw='.sha1($this->user_pw);
		
		$this->init_login();
		
		return $this->login_check();
		*/
		
		return $this->login($user_id, $user_pw);
	}
	
	/**
	 * Gibt Account-Informationen zurck
	 * 
	 * @return array
	 */
	public function get_account_info()
	{
		$postdata 	= array('uid' => $this->user_id,
							'upw' => $this->user_pw);
		$res		= $this->get_curl('http://uploaded.to/status', $postdata);
		$a 			= array();

		if(preg_match('#blocked#i', $res))
		{
			return array('err' => 'Gesperrt! 15min Warten');	
		}
		
		if(preg_match('#wrong user#i', $res))
		{
			return array('err' => 'falsche UserID');
		}
		
		if(preg_match('#wrong password#i', $res))
		{
			return array('err' => 'falsches Passwort');
		}
		
		if(!preg_match('#status: premium#i', $res))
		{
			return array('acc_status' => 'Free Account');
		}
		
		preg_match('#traffic: (\d+)#i', $res, $matches);
		$a['traffic'] = $matches[1];
		
		preg_match('#expire: (\d+)#i', $res, $matches);
		$a['expire'] = $matches[1];
		
		$a['acc_status'] = 'Premium Account';
		
		return $a;
		
	}

	/**
	 * Ld eine Datei auf Uploaded.to hoch
	 * 
	 * @param string $file Pfad zur Datei z.B. C:\text.txt oder /home/text.txt
	 * 
	 * @return bool|array false = Upload fehlgeschlagen | array = ['id', 'editKey']
	 */
	public function upload($file)
	{		
		if(false === $rf = realpath($file))
			return false;
		
		$key 		= Uploaded::generate();
		$postdata 	= array('Filename' => basename($rf),
							'Upload' => 'Submit Query',
							'Filedata' => '@'.$rf.';type='.Uploaded::get_mime_type($rf));		
		$res		= $this->get_curl($this->get_upload_server().'upload?admincode='.$key.$this->auth_str, 
											$postdata);
		
		return ($res === false) ? false : array('id' => (strpos($res, ',') ? substr($res, 0, strpos($res, ',')) : $res), 
												'editKey' => $key);
	}
	
	/**
	 * Ermglicht das herunterladen einer Datei.
	 * Nur mit Premium Account 
	 * 
	 * @param string $file_id File-ID
	 * @param string $path Download-Verzeichnis
	 */
	public function download($file_id, $path)
	{
		if(false === $rf = realpath($path))
			return false;
		
		$file_header = $this->get_curl('http://uploaded.to/file/'.$file_id.'/ddl', null, true, true, 
										array(	CURLOPT_AUTOREFERER => true,
												CURLOPT_FOLLOWLOCATION => true,
												CURLOPT_NOBODY => true));
		
		if(!preg_match('#filename=\"(.*?)\"#mi', $file_header, $matches))
			return false;
			
		$file 	= $rf."/".$matches[1];
		
		if(file_exists($file))
			return false;

		$fp		= fopen($file, 'w+');		
		$res	= $this->get_curl('http://uploaded.to/file/'.$file_id.'/ddl', null, true, false,
										array(	CURLOPT_AUTOREFERER => true,
												CURLOPT_FOLLOWLOCATION => true,
												CURLOPT_FILE => $fp));	
		fclose($fp);
		
		return $res;
	}
	
	/**
	 * Verschiebt eine Datei in einen Ordner
	 * 
	 * @param string $file_id		File-ID
	 * @param string $folder_id		Ordner-ID
	 */
	public function move_to_folder($file_id, $folder_id)
	{
		return $this->get_curl('http://uploaded.to/io/me/folder/'.$folder_id.'/add/'.$file_id) === '';
	}
	
	/**
	 * Benennt einen Ordner um
	 * 
	 * @param string $folder_id
	 * @param string $folder_name
	 * 
	 * @return bool
	 */
	public function set_folder_name($folder_id, $folder_name)
	{
		return $this->set_name('folder', $folder_id, $folder_name);
	}
	
	/**
	 * Erstellt einen neuen Ordner
	 * 
	 * @param string $folder_name
	 * 
	 * @return string Ordner-ID
	 */
	public function create_folder($folder_name)
	{
		$this->get_curl('http://uploaded.to/io/me/folder/create');
		$id = $this->get_folders('title', 'Neuer Ordner', 'id');
		$this->set_folder_name($id, $folder_name);
		
		return $id;
	}
	
	/**
	 * Umbennen einer Datei
	 * 
	 * @param string $file_id
	 * @param string $file_name
	 * 
	 * @return bool
	 */
	public function set_file_name($file_id, $file_name)
	{
		return $this->set_name('files', $file_id, $file_name);
	}
	
	/**
	 * Lscht einen Ordner
	 * 
	 * @param string $folder_id
	 * 
	 * @return bool
	 */
	public function delete_folder($folder_id)
	{
		return $this->delete_obj('folder', $folder_id);
	}
	
	/**
	 * Lscht eine Datei
	 * 
	 * @param string $file_id
	 * 
	 * @return bool
	 */
	public function delete_file($file_id)
	{
		return $this->delete_obj('files', $file_id);
	}
	
	/**
	 * Gibt ein oder mehrere Folder-Objekte zurck, einzeln oder in einem Array.
	 * 
	 * @param string $filer_type [optional]  Auf was soll gefiltert werden [files, id, ispublic, title]
	 * @param string $filter [optional] Wert des Filters
	 * @param int|string $limit_opt [optional] 0 = Gibt Folder-Objekt zurck | >0 = Gibt limitierte Anzahl von Folder-Objekt in Array zurck | String = gibt Folder-Objekt Eigenschaft zurck
	 * 
	 * @return mixed false = Fehler | array = Enthlt ein oder mehrere Folder-Objekte | Objekt = Folder-Objekt | String = Wert einer Folder-Objekt Eigenschaft
	 */
	public function get_folders($filer_type = '', $filter = '', $limit_opt = false)
	{
		return $this->get_list_obj('folders', $filer_type, $filter, $limit_opt);
	}
	
	/**
	 * Gibt ein oder mehrere File-Objekte zurck, einzeln oder in einem Array.
	 * 
	 * @param string $filer_type [optional]  Auf was soll gefiltert werden [admin, available, date, ddl, desc, dls, file_extension, filename, id, lastdownload, privacy, size]
	 * @param string $filter [optional] Wert des Filters
	 * @param int|string $limit_opt [optional] 0 = Gibt File-Objekt zurck | >0 = Gibt limitierte Anzahl von File-Objekt in Array zurck | String = gibt File-Objekt Eigenschaft zurck
	 * 
	 * @return mixed false = Fehler | array = Enthlt ein oder mehre File-Objkete | Objekt = File-Objekt | String = Wert einer File-Objekt Eigenschaft
	 */
	public function get_files($filer_type = '', $filter = '', $limit_opt = false)
	{
		return $this->get_list_obj('files', $filer_type, $filter, $limit_opt);
	}
	
	/**
	 * Sicherheitskopie einer Datei anlegen
	 * 
	 * @param String $file_id
	 * @return boolean|mixed
	 */
	public function backup_file($file_id)
	{
		
		$json = $this->get_curl('http://uploaded.to/io/file/backup/'.$file_id);
		$obj = json_decode(str_replace(array('auth', 'filename', 'size'), array('"auth"', '"filename"', '"size"'), $json));
			
		if($obj === null || isset($obj->err))
			return false;
		
		return $obj;
	}
	
	/**
	 * Startet ein Remote Upload
	 * 
	 * @param string $urls Remote URL (1 URL)
	 */
	public function add_remote_upload($url)
	{
		return $this->get_curl('http://uploaded.to/io/remote/add', 'values='.urlencode($url)) === '<em class=\'cG\'>Added</em>: '.$url."\n";	
	}
	
	protected function set_name($type, $obj_id, $obj_name)
	{	
		return $this->get_curl('http://uploaded.to/io/me/'.$type.'/'.$obj_id.'/set/title', 'value='.urlencode($obj_name).'&editorId=') == $obj_name;
	}
	
	protected function delete_obj($type, $obj_id)
	{
		return  $this->get_curl('http://uploaded.to/io/me/'.$type.'/'.$obj_id.'/delete') == '';
	}
	
	protected function get_list_obj($list_type, $filer_type = '', $filter = '', $limit_opt = false)
	{
		switch($list_type)
		{
			case 'folders':
				$obj = json_decode($this->get_curl('http://uploaded.to/io/me/list/folders', 
													'page=0'));
				break;
				
			default:
				$obj = json_decode($this->get_curl('http://uploaded.to/io/me/list/files', 
													'page=0&limit=100&order=date&dir=desc&search='));		
		}
				
		if($obj === null || isset($obj->err))
			return false;
		
		if($filter && $filer_type)
		{
			$files	= array();
			$i		= 0;
			$limit	= $limit_opt;
			$opt	= false;
			
			if($limit_opt !== false && !is_numeric($limit_opt))
			{
				$limit = 0;
				$opt = $limit_opt;
			}
									
			foreach($obj->list as $list)
			{
				if(!isset($list->$filer_type) || $list->$filer_type != $filter)
					continue;
				
				if($limit === $i++)
					return ($limit === 0) ? ($opt !== false && isset($list->$opt)) 
											? $list->$opt
											: $list
										: $files;
				
				$files[] = $list;
			}
			
			return $i ? $files : false;
		}
				
		return $obj->list;
	}
		
	protected function get_curl($url, $post = null, $cookie = true, $header = false, $opt = null)
	{
		$ch = curl_init();
						
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		
		if($post !== null)
		{
			curl_setopt($ch, CURLOPT_POST, true);
			curl_setopt($ch, CURLOPT_POSTFIELDS, $post);	
		}
		
		if($cookie)
			curl_setopt($ch, CURLOPT_COOKIE, $this->cookie_str);
		
		if($header)
			curl_setopt($ch, CURLOPT_HEADER, true);
		
		if($opt !== null)
		{
			curl_setopt_array($ch, $opt);
		}
				
		$res = curl_exec($ch);
		
		curl_close($ch);
		
		return $res;
	}
	
	protected function get_upload_server()
	{
		$res = $this->get_curl('http://uploaded.to/js/script.js');
		
		if($res !== false && preg_match('#uploadServer = \'(.*?)\';#', $res, $machtes))
			return $machtes[1];
		
		return false;
	}
	
	protected static function generate($len = 6)
	{
		$pwd = '';
		$con = array('b','c','d','f','g','h','j','k','l','m','n','p','r','s','t','v','w','x','y','z');
		$voc = array('a','e','i','o','u');
		$len /= 2;
		
		for($i = 0; $i < $len; $i++)
		{
			$c = mt_rand(0, 1000) % 20;
			$v = mt_rand(0, 1000) % 5;
			$pwd .= $con[$c].$voc[$v];
		}
		
		return $pwd;
	}
	
	protected static function get_mime_type($file)
	{
		$mime_type = '';
		
		if(function_exists('finfo_file'))
		{
			$finfo 		= finfo_open(FILEINFO_MIME_TYPE);
			$mime_type 	= finfo_file($finfo, $file);
			
			finfo_close($finfo);
		}
		elseif(function_exists('mime_content_type'))
		{
			$mime_type =  mime_content_type($file);
		}
		
		return $mime_type;
	}
	
	public function __sleep()
	{
		return array('user_id', 'user_pw', 'auth_str', 'cookie_str');	
	}
	
	public function __wakeup()
	{
			$this->init_login();
	}
}
?>